'use strict';
var express = require('express');
var path = require('path');
var cookieParser = require('cookie-parser');
var bodyParser = require('body-parser');
var debug = require('debug')('app');
var compress = require('compression');
var swaggerTools = require('swagger-tools');
var http = require('http');
var util = require('./routes/util');
const { exec } = require('child_process');

var app = express();

process.on('uncaughtException', function(err) {
	console.log('uncaughtException: ' + err);
});

app.set('restPort', process.env.PORT || 3000);
app.server = require('http').Server(app).listen(app.get('restPort'), function(){debug(JSON.stringify(app.server.address())); });
app.server.timeout = 0; // disable timeout on incoming connections

//trust local Apache gateway proxy which dispatching all http requests to FM server
app.set('trust proxy', 'loopback');
app.enable('case sensitive routing');

//remove 'x-powered-by: express' header
app.disable('x-powered-by');

app.use(compress());
app.use(bodyParser.json({limit: '25mb'}));
app.use(bodyParser.urlencoded({limit: '25mb', extended: true}));
app.use(cookieParser());

if (util.cloudVersion() === 2) {
	app.use('/fmi/data/v1', express.static(path.join(__dirname, 'public-cloud')));
	app.use('/fmi/data/vLatest', express.static(path.join(__dirname, 'public-cloud')));
} else {
	app.use('/fmi/data/v1', express.static(path.join(__dirname, 'public')));
	app.use('/fmi/data/vLatest', express.static(path.join(__dirname, 'public')));
}
if (util.cloudVersion() === 1) {
	app.use('/fmi/data/apidoc/', express.static(path.resolve(__dirname,'public/docs')));
} else if (util.cloudVersion() === 2) {
	app.use('/fmi/data/apidoc/', express.static(path.resolve(__dirname,'public-cloud/docs')));
} else {
	app.use('/fmi/data/apidoc/', express.static(path.resolve(__dirname,'../../../Documentation/Data\ API\ Documentation')));
}

var swaggerDoc;

if (util.cloudVersion() === 2) {
	swaggerDoc = require('./dapi_v1.cloud.json');
 } else {
	swaggerDoc = require('./dapi_v1.json');
 }

// swaggerRouter configuration
var options = {
	controllers: path.join(__dirname, 'controllers'),
	useStubs: false
};

var validatorOptions = {
	validateResponse: false
};

// Initialize the Swagger middleware
swaggerTools.initializeMiddleware(swaggerDoc, function (middleware) {
	// Interpret Swagger resources and attach metadata to request - must be first in swagger-tools middleware chain
	app.use(middleware.swaggerMetadata());
  
	// Validate Swagger requests
	app.use(middleware.swaggerValidator(validatorOptions));
  
	// Route validated requests to appropriate controller
	app.use(middleware.swaggerRouter(options));
  
	// default error handler
	app.use(function(err, req, res, next) {
		if (res.headersSent) {
			return next(err)
		}
		var error = new Error();
		if (err.failedValidation) {
			if (err.results && err.results.errors) {
				var e = err.results.errors[0];
				if (e.code == "OBJECT_MISSING_REQUIRED_PROPERTY") {
					error.messages = [{"message": e.message, "code":"10"}];
				} else {
					error.messages = [{"message": (e.path) ? "'" + e.path[0] + "': " + e.message : e.message, "code":"960"}];
				}
			} else {
				if (err.code === "REQUIRED") {
					error.messages = [{"message": err.message, "code":"10"}];
				} else if (err.message.startsWith('Invalid content type')) {
					error.messages = [{"message": err.message, "code":"1708"}];
				}
			}
			error.response = {};
			res.set('Content-Type', 'application/json');
			res.end(JSON.stringify(error));
	} else {
		if (err.allowedMethods && -1 === err.allowedMethods.indexOf(req.method.toUpperCase())) {
			error.messages = [{"message":"Resource doesn't support the specified HTTP verb", "code":"1704"}];
		} else {
			error.messages = [{"message":err.message, "code":"1630"}];
		}
		error.response = {};
		res.set('Content-Type', 'application/json');
		res.end(JSON.stringify(error));
		}
	});
});

var licenseApi = require('./routes/api_license');
app.use('/fmi/data/vLatest/refreshLicense', licenseApi.router);
var cursorApi = require('./routes/api_cursor');
app.use('/fmi/data/:version/databases/:solution/layouts/:layout/cursor', function (req, res, next) {
	res.locals.version = req.params.version;
	res.locals.solution = req.params.solution
	res.locals.layout = req.params.layout;
	res.locals.cursorToken = req.get('X-FM-Data-Cursor-Token');
	var auth = req.get('Authorization');
	if (auth.toLowerCase().startsWith("bearer ")) {
		res.locals.token = auth.split(" ")[1];
	} else {
		return errorHandler.handleError('Unauthorized', req, res, "HTTP Authorization header is missing.","10");
	}
	next();
});
app.use('/fmi/data/:version/databases/:solution/layouts/:layout/cursor', cursorApi.router);

if (util.cloudVersion() === 2) {
	app.use('/fmi/data/vLatest/fmidurl', function (req, res, next) {
		res.set('Content-Type', 'application/json');
		http.get({
			hostname:'localhost',
			port: '1895',
			path: '/oauth/fmidurl',
			headers: {
				'x-fms-return-url': 'https://' + req.get("X-Forwarded-Host") + '/fmi/data/vLatest/tableau/fm_connector.html'
			}
		}, (http_res) => {
			let data = '';
			http_res.on('data', (chunk) => { data += chunk; })
			http_res.on('end', () => {
				try {
					res.end('{ "url": "' + data + '" }');
				} catch (e) {

				}
			});
		});
	});
	app.use('/fmi/data/vLatest/apihost', function (req, res, next) {
		exec('"/opt/FileMaker/FileMaker Server/Database Server/bin/fmscldauth" --bu', (error, stdout, stderr) => {
			res.end('{ "API_Host": "' + stdout.trim() + '"}');
		});
	});
	app.use('/fmi/data/vLatest/tokenexchange', function (req, res, next) {
		res.set('Content-Type', 'application/json');
		const body = 'code=' + req.body.code + '&redirectUrl=http://localhost/fmi/dummy';
		const r = http.request({
			hostname:'localhost',
			port: '16002',
			path: '/fmsadminapi/fmid/tokenexchange',
			method: 'POST',
			headers: {
				'Content-Type':'application/x-www-form-urlencoded',
				'Content-Length': Buffer.byteLength(body)
			}
		}, (http_res) => {
			let data = '';
			http_res.on('data', (chunk) => { data += chunk; })
			http_res.on('end', () => {
				let resp = JSON.parse(data);
				try {
					res.end('{ "token": "' + resp.jwt + '", "refresh": "' + resp.refresh + '" }');
				} catch (e) {

				}
			});
		});
		r.write(body);
		r.end();
	});
}

/// error handlers
// app.use(function(err, req, res, next) {
// 	res.status(err.status || 500);
// 	res.send(err.message);
// 	res.end();
// });

process.on('SIGUSR1', function() {
	console.log('Stop the process');

	// add clean up code to stop the server
	app.server.close();
	process.exit(0);
});

module.exports = app;
module.exports.premises = util.checkPremises();
